/*
** nester - NES emulator
** Copyright (C) 2000  Darren Ranalli
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
** Library General Public License for more details.  To obtain a 
** copy of the GNU Library General Public License, write to the Free 
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** Any permitted reproduction of these routines, in whole or in part,
** must bear this legend.
*/

#ifndef _NES_MAPPER_H_
#define _NES_MAPPER_H_

#include "../NES_ROM.h"
#include "../../debug/debug.h"

#define MAPPERINLINE __inline

/////////////////////////////////////////////////////////////////////
// mapper factory
/////////////////////////////////////////////////////////////////////

typedef struct nesMapper_s
{
  void (*FakeConstructor)();

  void  (*Reset)();

  void  (*MemoryWrite)(uint32 addr, uint8 data);
  void  (*MemoryWriteLow)(uint32 addr, uint8 data);
  void  (*MemoryWriteSaveRAM)(uint32 addr, uint8 data);

  void  (*HSync)(uint32 scanline);
  void  (*VSync)();

  // for mmc2
  void  (*PPU_Latch_FDFE)(uint32 addr);

  void (*set_CPU_banks)(uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void (*set_CPU_bank4)(uint32 bank_num);
  void (*set_CPU_bank5)(uint32 bank_num);
  void (*set_CPU_bank6)(uint32 bank_num);
  void (*set_CPU_bank7)(uint32 bank_num);

  // for mapper 40
  void (*set_CPU_banksM40)(uint32 bank3_num,
                     uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void (*set_CPU_bank3)(uint32 bank_num);

  void (*set_PPU_banks)(uint32 bank0_num, uint32 bank1_num,
                     uint32 bank2_num, uint32 bank3_num,
                     uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void (*set_PPU_bank0)(uint32 bank_num);
  void (*set_PPU_bank1)(uint32 bank_num);
  void (*set_PPU_bank2)(uint32 bank_num);
  void (*set_PPU_bank3)(uint32 bank_num);
  void (*set_PPU_bank4)(uint32 bank_num);
  void (*set_PPU_bank5)(uint32 bank_num);
  void (*set_PPU_bank6)(uint32 bank_num);
  void (*set_PPU_bank7)(uint32 bank_num);

  void (*set_mirroring)(uint32 nt0, uint32 nt1, uint32 nt2, uint32 nt3);
  void (*set_mirroringB)(mirroring_type m);
} nesMapper_t;
nesMapper_t* GetMapper();

/////////////////////////////////////////////////////////////////////
// Mapper virtual base class
  void NES_mapper_FakeConstructor();

  static MAPPERINLINE void  NES_mapper_Reset() {};

  static MAPPERINLINE void  NES_mapper_MemoryWrite(uint32 addr, uint8 data)        {}
  static MAPPERINLINE void  NES_mapper_MemoryWriteLow(uint32 addr, uint8 data)     {}
  static MAPPERINLINE void  NES_mapper_MemoryWriteSaveRAM(uint32 addr, uint8 data) {}

  static MAPPERINLINE void  NES_mapper_HSync(uint32 scanline)  {}
  static MAPPERINLINE void  NES_mapper_VSync()  {}

  // for mmc2
  static MAPPERINLINE void  NES_mapper_PPU_Latch_FDFE(uint32 addr) {}

  void NES_mapper_set_CPU_banks(uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void NES_mapper_set_CPU_bank4(uint32 bank_num);
  void NES_mapper_set_CPU_bank5(uint32 bank_num);
  void NES_mapper_set_CPU_bank6(uint32 bank_num);
  void NES_mapper_set_CPU_bank7(uint32 bank_num);

  // for mapper 40
  void NES_mapper_set_CPU_banksM40(uint32 bank3_num,
                     uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void NES_mapper_set_CPU_bank3(uint32 bank_num);

  void NES_mapper_set_PPU_banks(uint32 bank0_num, uint32 bank1_num,
                     uint32 bank2_num, uint32 bank3_num,
                     uint32 bank4_num, uint32 bank5_num,
                     uint32 bank6_num, uint32 bank7_num);
  void NES_mapper_set_PPU_bank0(uint32 bank_num);
  void NES_mapper_set_PPU_bank1(uint32 bank_num);
  void NES_mapper_set_PPU_bank2(uint32 bank_num);
  void NES_mapper_set_PPU_bank3(uint32 bank_num);
  void NES_mapper_set_PPU_bank4(uint32 bank_num);
  void NES_mapper_set_PPU_bank5(uint32 bank_num);
  void NES_mapper_set_PPU_bank6(uint32 bank_num);
  void NES_mapper_set_PPU_bank7(uint32 bank_num);

  void NES_mapper_set_mirroring(uint32 nt0, uint32 nt1, uint32 nt2, uint32 nt3);
  void NES_mapper_set_mirroringB(mirroring_type m);

/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 0
  void  NES_mapper0_Reset();
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 1
  void  NES_mapper1_Reset();

  void  NES_mapper1_MemoryWrite(uint32 addr, uint8 data);
  // this uses MMC1_256K_base and MMC1_bankX
  void NES_mapper1_MMC1_set_CPU_banks();
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 2
  void  NES_mapper2_Reset();

  void  NES_mapper2_MemoryWrite(uint32 addr, uint8 data);
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 3
  void  NES_mapper3_Reset();

  void  NES_mapper3_MemoryWrite(uint32 addr, uint8 data);
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 4
extern uint8  regs[8];
extern uint32 prg0,prg1;
extern uint32 chr01,chr23,chr4,chr5,chr6,chr7;
extern uint32 num_8k_ROM_banks;
extern uint32 num_1k_VROM_banks;
//uint32 chr_swap();
//uint32 prg_swap();

  void  NES_mapper4_Reset();

  void  NES_mapper4_MemoryWrite(uint32 addr, uint8 data);

  void  NES_mapper4_HSync(uint32 scanline);

  static MAPPERINLINE uint32 chr_swap() { return (regs[0] & 0x80); }
  static MAPPERINLINE uint32 prg_swap() { return (regs[0] & 0x40); }

  static MAPPERINLINE void NES_mapper4_MMC3_set_CPU_banks()
  {
    if(prg_swap())
    {
      NES_mapper_set_CPU_banks(num_8k_ROM_banks-2,prg1,prg0,num_8k_ROM_banks-1);
    }
    else
    {
      NES_mapper_set_CPU_banks(prg0,prg1,num_8k_ROM_banks-2,num_8k_ROM_banks-1);
    }
  }

  static MAPPERINLINE void NES_mapper4_MMC3_set_PPU_banks()
  {
    if(num_1k_VROM_banks)
    {
      if(chr_swap())
      {
        NES_mapper_set_PPU_banks(chr4,chr5,chr6,chr7,chr01,chr01+1,chr23,chr23+1);
      }
      else
      {
        NES_mapper_set_PPU_banks(chr01,chr01+1,chr23,chr23+1,chr4,chr5,chr6,chr7);
      }
    }
  }

  void NES_mapper4_SNSS_fixup(); // HACK HACK HACK HACK

/////////////////////////////////////////////////////////////////////
// Mapper 7
  void  NES_mapper7_Reset();

  void  NES_mapper7_MemoryWrite(uint32 addr, uint8 data);
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 9
  void  NES_mapper9_Reset();

  void  NES_mapper9_MemoryWrite(uint32 addr, uint8 data);

  void  NES_mapper9_PPU_Latch_FDFE(uint32 addr);

  void NES_mapper9_set_VROM_0000();
  void NES_mapper9_set_VROM_1000();

  void NES_mapper9_SNSS_fixup(); // HACK HACK HACK HACK

/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 40 (smb2j)
  void  NES_mapper40_Reset();

  void  NES_mapper40_MemoryWrite(uint32 addr, uint8 data);

  void  NES_mapper40_HSync(uint32 scanline);
/////////////////////////////////////////////////////////////////////

#include "NES_mapper_Konami.h"

/////////////////////////////////////////////////////////////////////

#endif
